<?php

namespace FwRoutingSystem;

use Api\BaseRouter;
use AuthMiddleware;
use Closure;
use Controller;
use ControllerScheme;
use DATABASE\ORM\QueryBuilder\QueryBuilder\Db;
use Exception;
use FwAuthSystem\Main\Auth;
use FwAuthSystem\Main\UserObject;
use FwAuthSystem\Utils\AuthConfig;
use FwRoutingSystem\Router\RouterCommand;
use FwRoutingSystem\Router\RouterException;
use FwRoutingSystem\Router\RouterRequest;
use ReflectionMethod;
use version\ApiVersions;

/**
 * Class Router
 *
 * @method $this any($route, $settings, $callback = NULL)
 * @method $this get($route, $settings, $callback = NULL)
 * @method $this post($route, $settings, $callback = NULL)
 * @method $this put($route, $settings, $callback = NULL)
 * @method $this delete($route, $settings, $callback = NULL)
 * @method $this patch($route, $settings, $callback = NULL)
 * @method $this head($route, $settings, $callback = NULL)
 * @method $this options($route, $settings, $callback = NULL)
 * @method $this xpost($route, $settings, $callback = NULL)
 * @method $this xput($route, $settings, $callback = NULL)
 * @method $this xdelete($route, $settings, $callback = NULL)
 * @method $this xpatch($route, $settings, $callback = NULL)
 *
 */
class Router {
	/**
	 * @var string $baseFolder Pattern definitions for parameters of Route
	 */
	public $baseFolder;
	public $outputSent = false;
	/**
	 * @var string
	 */
	protected $documentRoot = '';
	/**
	 * @var string
	 */
	protected $runningPath = '';
	/**
	 * @var array $routes Routes list
	 */
	protected $routes = [];
	/**
	 * @var array $groups List of group routes
	 */
	protected $groups = [];
	/**
	 * @var array $patterns Pattern definitions for parameters of Route
	 */
	protected $patterns = [
		':id'      => '(\d+)',
		':number'  => '(\d+)',
		':any'     => '([^/]+)',
		':all'     => '(.*)',
		':string'  => '(\w+)',
		'{s}'      => '(\w+)',
		':slug'    => '([\w\-_]+)',
		':persian' => '([ا-ی\w\-_]+)',
	];
	/**
	 * @var array $namespaces Namespaces of Controllers and Middlewares files
	 */
	protected $namespaces = [
		'middlewares' => '',
		'controllers' => '',
	];
	/**
	 * @var array $path Paths of Controllers and Middlewares files
	 */
	protected $paths = [
		'controllers' => 'controllers',
		'middlewares' => 'middlewares',
	];
	/**
	 * @var string $mainMethod Main method for controller
	 */
	protected $mainMethod = 'main';
	/**
	 * @var string $cacheFile Cache file
	 */
	protected $cacheFile = NULL;
	/**
	 * @var bool $cacheLoaded Cache is loaded?
	 */
	protected $cacheLoaded = false;
	/**
	 * @var Closure $errorCallback Route error callback function
	 */
	protected $errorCallback;
	/**
	 * @var array $middlewares General middlewares for per request
	 */
	protected $middlewares = [];
	/**
	 * @var array $routeMiddlewares Route middlewares
	 */
	protected $routeMiddlewares = [];
	/**
	 * @var array $middlewareGroups Middleware Groups
	 */
	protected $middlewareGroups = [];

	/**
	 * Router constructor method.
	 *
	 * @param array $params
	 *
	 * @return void
	 */
	public function __construct(array $params = [
		'paths'      => [
			'controllers' => 'controllers',
		],
		'namespaces' => [
			'controllers' => 'controller',
		],
	]) {
		$this->documentRoot = realpath($_SERVER['DOCUMENT_ROOT']);
		$this->runningPath = realpath(getcwd());
		$this->baseFolder = $this->runningPath;
		if (isset($params['debug']) && is_bool($params['debug'])) {
			RouterException::$debug = $params['debug'];
		}
		$this->setPaths($params);
		$this->loadCache();
	}

	/**
	 * Set paths and namespaces for Controllers and Middlewares.
	 *
	 * @param array $params
	 *
	 * @return void
	 */
	protected function setPaths($params) {

		if (empty($params)) {
			return;
		}
		if (isset($params['paths']) && $paths = $params['paths']) {
			$this->paths['controllers'] = isset($paths['controllers'])
				? trim($paths['controllers'], '/')
				: $this->paths['controllers'];
			$this->paths['middlewares'] = isset($paths['middlewares'])
				? trim($paths['middlewares'], '/')
				: $this->paths['middlewares'];
		}
		if (isset($params['namespaces']) && $namespaces = $params['namespaces']) {
			$this->namespaces['controllers'] = isset($namespaces['controllers'])
				? trim($namespaces['controllers'], '\\') . '\\'
				: '';
			$this->namespaces['middlewares'] = isset($namespaces['middlewares'])
				? trim($namespaces['middlewares'], '\\') . '\\'
				: '';
		}
		if (isset($params['base_folder'])) {
			$this->baseFolder = rtrim($params['base_folder'], '/');
		}
		if (isset($params['main_method'])) {
			$this->mainMethod = $params['main_method'];
		}
		$this->cacheFile = isset($params['cache']) ? $params['cache'] : realpath(__DIR__ . '/../cache.php');
	}

	/**
	 * Load Cache file
	 *
	 * @return bool
	 */
	protected function loadCache() {
		if (file_exists($this->cacheFile)) {
			$this->routes = require $this->cacheFile;
			$this->cacheLoaded = true;
			return true;
		}
		return false;
	}

	public function __defaults() {
		$this->get('/$|/index', function () {
			if (UserObject::instance() instanceof UserObject) {
				include __SOURCE__ . 'index.php';
			} else {
				header("Location: /admin/login");
			}
		});
		$this->group('/', function (Router $router) {
			$router->any('/api/profile/:string/:string', function ($controller, $method) {
				$controller = "\controller\\$controller";
				if (class_exists('\controller\\Customers')) {
					$class = new $controller();
					if ($class instanceof ControllerScheme) {
						if ($class->isApi()) {
							if (method_exists($class, $method)) {
								$class->profile = true;
								$class->{$method}();
								echo json_encode(array_map(function ($item) {
									return "";
								}, $class->params), JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
							} else {
								return "Method Not Found";
							}
						} else {
							return "Class Is Not An Api";
						}
					} else {
						return "Class Is Not A ControllerScheme";
					}
				} else {
					return "Class Not Found";
				}
			});
			$router->any('/modules/:string/:any?/:any?/:any?/:any?/:any?:any?', function () {
				$str = implode(DIRECTORY_SEPARATOR, func_get_args());
				$host = $_SERVER['HTTP_HOST'];
				$referer = urldecode($_SERVER['HTTP_REFERER']);
				$referer = str_replace("http://", "", str_replace($host, "", str_replace("https://", "", $referer)));
				$referer = str_replace("/src/", "", $referer);
				$referer = str_replace("src/", "", $referer);
				$referer = strtok($referer, '?');
//		var_dump($referer);
//		if (file_exists(__SOURCE__ . $referer)) {
				header("Content-Type: text/javascript");
				if (!str($str)->endsWith('.js')) $str .= '.js';
				if (file_exists(__SOURCE__ . 'modules/' . $str)) {
					include __SOURCE__ . 'modules/' . $str;
				} else {
					include __SOURCE__ . 'modules/notFoundModule.js';
				}
//		} else {
//			return "request failed";
//		}
			});
			$router->any('/conf/Activate', function () {
				$db = Db::table('tblActiveList');
				if ($_POST['action'] == 'activate') {
					if (!$db->where([
						'item_id'    => $_POST['item_id'],
						'table_name' => $_POST['table_name'],
					])->get()->first()) {
						$res = $db->insert([
							'item_id'    => $_POST['item_id'],
							'table_name' => $_POST['table_name'],
							'date'       => time(),
							'user_id'    => UserObject::instance()->getUserId(),
						]);
						if ($res) {
							return 1;
						} else {
							return 0;
						}
					} else {
						return 500;
					}
				} else {
					if ($db->where([
						'item_id'    => $_POST['item_id'],
						'table_name' => $_POST['table_name'],
					])->get()->first()) {
						$res = $db->where([
							'item_id'    => $_POST['item_id'],
							'table_name' => $_POST['table_name'],
						])->delete();
						if ($res) {
							return 2;
						} else {
							return 0;
						}
					} else {
						return 404;
					}
				}
			});
			$router->any('/js/:all', function ($param) {
				header("Content-Type: text/javascript");
				$param = strtok($param, "?");
				$text = file_get_contents(__SOURCE__ . 'views/' . $param);
				$params = explode('/', $param);
				$last = end($params);
				array_pop($params);
				$last = strtok($last, '.');
				$path = "controllers/" . implode('/', $params) . '/' . $last;
				$pos = strpos($text, 'import');
				$newText = substr($text, $pos, strlen($text));
				echo substr_replace($text, ";\nconst CurrentPathForController = '{$path}'\n", stripos($newText, ';'), 0);
			});
			$router->any('controllers/:all', function (string $path) {
				if (UserObject::instance() instanceof UserObject) {
					if (is_file(__SOURCE__ . "controllers/$path.php")) {
						$ClassName = 'controller\\' . collect(str($path)->explode('/'))->last();
						if (class_exists($ClassName)) {
							$Class = new $ClassName();
							if ($Class instanceof ControllerScheme) {
								return $Class->do();
							}
						}
					} else {
						if (str($path)->includes('@')) {

							$classArr = explode('@', $path);
						} else {
							$classArr = [
								$path,
								'main'
							];
						}
						$ClassName = 'controller\\' . $classArr[0];
						if (class_exists($ClassName)) {
							$Class = new $ClassName();
							if ($Class instanceof ControllerScheme) {
								return $Class->do($classArr[1]);
							}
						}
					}
					return '404';
				} else {
					return '403';
				}
			});
			$router->any('views/.{1,}/:string?', ['before' => new AuthMiddleware()], function () {
				$request_url = $_SERVER['REQUEST_URI'];
				if (strpos($request_url, '?')) {
					$request_url = explode('?', $request_url)[0];
				}
				$remove = trim(str_replace($_SERVER['HTTP_HOST'], '', __HOST__));
				$request_url = str_replace($remove, '', $request_url);
				if (file_exists(__SOURCE__ . str_replace('/src/', '', ($request_url)) . '.php') and !is_dir(__SOURCE__ . str_replace('/src/', '', ($request_url)) . '.php')) {
					include __SOURCE__ . str_replace('/src/', '', ($request_url)) . '.php';
				}
			});
			$router->any('fwTools/controller/.{1,}/:string?', function () {
				$request_url = $_SERVER['REQUEST_URI'];
				if (strpos($request_url, '?')) {
					$request_url = explode('?', $request_url)[0];
				}
				$remove = trim(str_replace($_SERVER['HTTP_HOST'], '', __HOST__));

				$request_url = str_replace($remove, '', $request_url);
				$request_url = str_replace('/admin/', '', $request_url);

				if (file_exists(__SOURCE__ . str_replace('/src/', '', ($request_url)) . '.php') and !is_dir(__SOURCE__ . str_replace('/src/', '', ($request_url)) . '.php')) {
					include __SOURCE__ . str_replace('/src/', '', ($request_url)) . '.php';
				}
			});
			$router->any('fwTools/view/.{1,}/:string?', function () {

				$request_url = $_SERVER['REQUEST_URI'];
				if (strpos($request_url, '?')) {
					$request_url = explode('?', $request_url)[0];
				}
				$remove = trim(str_replace($_SERVER['HTTP_HOST'], '', __HOST__));
				$request_url = str_replace($remove, '', $request_url);
				$request_url = str_replace('/admin/', '', $request_url);

				if (file_exists(__SOURCE__ . str_replace('/src/', '', ($request_url)) . '.php') and !is_dir(__SOURCE__ . str_replace('/src/', '', ($request_url)) . '.php')) {

					include __SOURCE__ . str_replace('/src/', '', ($request_url)) . '.php';
				}
			});
			$router->any('/\w{1,}\@\w{1,}', function () {
				header('location : https://' . $_SERVER['HTTP_HOST'] . collect(str($_SERVER['REQUEST_URI'])->explode('@'))->first);
			});
			$router->any('/[a-zA-Z\d]{1,}/API/:string?/_{s}?', function ($s = '') {
				$arr = explode('_', $s);
				$version = $arr[0];
				$controller_type = $arr[1];
				$request_url = $_SERVER['REQUEST_URI'];
				if (strpos($request_url, '?')) {
					$request_url = explode('?', $request_url)[0];
				}
				$filename = str_replace('src/', '', $request_url);
				$filename = explode('/API', $filename)[0];
				if ($filename[0] === '/') {
					$filename = substr($filename, 1, strlen($filename));
				}
				$arr = explode('/', $filename);
				$filename = end($arr);
				if (class_exists("\controller\\$filename")) {
					$filename = "\controller\\$filename";
					$class = (new $filename());
					$arr = (explode('API', $request_url));
					$controller_type = explode('?', $controller_type)[0];
					if ($class instanceof Controller and $class->isApi()) {
						$class->Api($controller_type, $version);
						return;
					}

				} else {
					return 'Not Found!';
				}
			});
			$router->any('/[a-zA-Z\d]{1,}', ['before' => new AuthMiddleware()], function () {

				if (endsWith($_SERVER['REQUEST_URI'], '/')) {
					$arr = str_split($_SERVER['REQUEST_URI']);
					unset($arr[sizeof($arr) - 1]);
					$arr = implode('', $arr);
					header('location: ' . $arr);
				}
				include __SOURCE__ . 'index.php';
			});
		});
		$this->any('/sign-out', function () {
			Auth::end();
			header('location: /'
			);
		});
		$this->error(function () {
			include __SOURCE__ . 'pages/404.php';
		});

		$this->get('/login', function () {
			$Auth = Auth::init(new AuthConfig());
			include __SOURCE__ . 'login.php';
		});

		$this->post('/login', function () {
			$Auth = Auth::init(new AuthConfig());
			include __SOURCE__ . 'login.php';
			$Auth->ProccessOnSubmit();
//			if ($_REQUEST['captcha'] == $_SESSION['captcha']) {
//				include __SOURCE__ . 'login.php';
//				$Auth->ProccessOnSubmit();
//			} else {
//				echo showErrorMsg("شما","اعتبار سنجی");
//				include __SOURCE__ . 'login.php';
//
//			}
		});
		$this->get('/captcha', function () {
			$captcha = generateRandomString(4, false);
			$_SESSION['captcha'] = md5($captcha);
			// Adapted for The Art of Web: www.the-art-of-web.com
			// Please acknowledge use of this code by including this header.
			// initialise image with dimensions of 160 x 45 pixels
			$image = @imagecreatetruecolor(160, 45) or die("Cannot Initialize new GD image stream");
			// set background and allocate drawing colours
			$background = imagecolorallocate($image, 0x66, 0xCC, 0xFF);
			imagefill($image, 0, 0, $background);
			$linecolor = imagecolorallocate($image, 0x33, 0x99, 0xCC);
			$textcolor1 = imagecolorallocate($image, 0x00, 0x00, 0x00);
			$textcolor2 = imagecolorallocate($image, 0xFF, 0xFF, 0xFF);
			// draw random lines on canvas
			for ($i = 0; $i < 8; $i++) {
				imagesetthickness($image, rand(1, 3));
				imageline($image, rand(0, 160), 0, rand(0, 160), 45, $linecolor);
			}
			// using a mixture of TTF fonts
			$fonts = [];
			$fonts[] = __SOURCE__ . '/dist/fonts/Vazir.ttf';
			$fonts[] = __SOURCE__ . '/dist/fonts/Vazir-Bold.ttf';
			$fonts[] = __SOURCE__ . '/dist/fonts/Vazir-Light.ttf';
			// add random digits to canvas using random black/white colour
			$digit = '';
			for ($x = 10; $x <= 130; $x += 30) {
				$textcolor = (rand() % 2) ? $textcolor1 : $textcolor2;
				$digit .= ($num = rand(0, 9));
				imagettftext($image, 20, rand(-30, 30), $x, rand(20, 42), $textcolor, $fonts[array_rand($fonts)], $num);
			}
			// record digits in session variable
			$_SESSION['captcha'] = $digit;
			// display image and clean up
			header('Content-type: image/png');
			imagepng($image);
			imagedestroy($image);
		});
	}

	/**
	 * Routes Group
	 *
	 * @param string $name
	 * @param closure|array $settings
	 * @param null|closure $callback
	 *
	 * @return bool
	 */
	public function group($name, $settings = NULL, $callback = NULL) {
		if ($this->cacheLoaded) {
			return true;
		}
		$group = [];
		$group['route'] = $this->clearRouteName($name);
		$group['before'] = $group['after'] = NULL;
		if (is_null($callback)) {
			$callback = $settings;
		} else {
			$group['before'][] = !isset($settings['before']) ? NULL : $settings['before'];
			$group['after'][] = !isset($settings['after']) ? NULL : $settings['after'];
		}
		$groupCount = count($this->groups);
		if ($groupCount > 0) {
			$list = [];
			foreach ($this->groups as $key => $value) {
				if (is_array($value['before'])) {
					foreach ($value['before'] as $k => $v) {
						$list['before'][] = $v;
					}
					foreach ($value['after'] as $k => $v) {
						$list['after'][] = $v;
					}
				}
			}
			if (isset($group['before']) and $group['after']) {
				if (!is_null($group['before'])) {
					$list['before'][] = $group['before'][0];
				}
				if (!is_null($group['after'])) {
					$list['after'][] = $group['after'][0];
				}
				$group['before'] = $list['before'];
				$group['after'] = $list['after'];
			}
		}
		$group['before'] = array_values(array_unique((array)$group['before']));
		$group['after'] = array_values(array_unique((array)$group['after']));
		array_push($this->groups, $group);
		if (is_object($callback)) {
			call_user_func_array($callback, [$this]);
		}
		$this->endGroup();
		return true;
	}

	/**
	 * @param string $route
	 *
	 * @return string
	 */
	private function clearRouteName($route = '') {
		$route = trim(str_replace('//', '/', $route), '/');
		return $route === '' ? '/' : "/{$route}";
	}

	/**
	 * Routes Group endpoint
	 *
	 * @return void
	 */
	private function endGroup() {
		array_pop($this->groups);
	}

	/**
	 * Routes error function. (Closure)
	 *
	 * @param $callback
	 *
	 * @return void
	 */
	public function error($callback) {
		$this->errorCallback = $callback;
	}

	/**
	 * [TODO] This method implementation not completed yet.
	 *
	 * @param string|array $middleware
	 *
	 * @return $this
	 */
	public function middlewareBefore($middleware) {
		$this->middleware($middleware, 'before');
		return $this;
	}

	/**
	 * [TODO] This method implementation not completed yet.
	 *
	 * Set route middleware
	 *
	 * @param string|array $middleware
	 * @param string $type
	 *
	 * @return $this
	 */
	public function middleware($middleware, $type = 'before') {
		if (!is_array($middleware) && !is_string($middleware)) {
			return $this;
		}
		$currentRoute = end($this->routes);
		$currentRoute[$type] = $middleware;
		array_pop($this->routes);
		array_push($this->routes, $currentRoute);
		return $this;
	}

	/**
	 * [TODO] This method implementation not completed yet.
	 *
	 * @param string|array $middleware
	 *
	 * @return $this
	 */
	public function middlewareAfter($middleware) {
		$this->middleware($middleware, 'after');
		return $this;
	}

	/**
	 * [TODO] This method implementation not completed yet.
	 *
	 * Set route name
	 *
	 * @param string $name
	 *
	 * @return $this
	 */
	public function name($name) {
		if (!is_string($name)) {
			return $this;
		}
		$currentRoute = end($this->routes);
		$currentRoute['name'] = $name;
		array_pop($this->routes);
		array_push($this->routes, $currentRoute);
		return $this;
	}

	/**
	 * [TODO] This method implementation not completed yet.
	 *
	 * Set general middlewares
	 *
	 * @param array $middlewares
	 *
	 * @return void
	 */
	public function setMiddleware(array $middlewares) {
		$this->middlewares = $middlewares;
	}

	/**
	 * [TODO] This method implementation not completed yet.
	 *
	 * Set Route middlewares
	 *
	 * @param array $middlewares
	 *
	 * @return void
	 */
	public function setRouteMiddleware(array $middlewares) {
		$this->routeMiddlewares = $middlewares;
	}

	/**
	 * [TODO] This method implementation not completed yet.
	 *
	 * Set middleware groups
	 *
	 * @param array $middlewareGroup
	 *
	 * @return void
	 */
	public function setMiddlewareGroup(array $middlewareGroup) {
		$this->middlewareGroups = $middlewareGroup;
	}

	/**
	 * Add route method;
	 * Get, Post, Put, Delete, Patch, Any, Ajax...
	 *
	 * @param $method
	 * @param $params
	 *
	 * @return mixed
	 * @throws
	 */
	public function __call($method, $params) {
		if ($this->cacheLoaded) {
			return true;
		}
		if (is_null($params)) {
			return false;
		}
		if (!in_array(strtoupper($method), explode('|', RouterRequest::$validMethods))) {
			return $this->exception($method . ' is not valid.');
		}
		$route = $params[0];
		$callback = $params[1];
		$settings = NULL;
		if (count($params) > 2) {
			$settings = $params[1];
			$callback = $params[2];
		}
		if (strstr($route, ':')) {
			$route1 = $route2 = '';
			foreach (explode('/', $route) as $key => $value) {
				if ($value != '') {
					if (!strpos($value, '?')) {
						$route1 .= '/' . $value;
					} else {
						if ($route2 == '') {
							$this->addRoute($route1, $method, $callback, $settings);
						}
						$route2 = $route1 . '/' . str_replace('?', '', $value);
						$this->addRoute($route2, $method, $callback, $settings);
						$route1 = $route2;
					}
				}
			}
			if ($route2 == '') {
				$this->addRoute($route1, $method, $callback, $settings);
			}
		} else {
			$this->addRoute($route, $method, $callback, $settings);
		}
		return $this;
	}

	/**
	 * Throw new Exception for Router Error
	 *
	 * @param $message
	 *
	 * @return RouterException
	 * @throws
	 */
	public function exception($message = '') {
		return new RouterException($message);
	}

	/**
	 * Add new Route and it's settings
	 *
	 * @param $uri
	 * @param $method
	 * @param $callback
	 * @param $settings
	 *
	 * @return void
	 */
	private function addRoute($uri, $method, $callback, $settings) {
		$groupItem = count($this->groups) - 1;
		$group = '';
		if ($groupItem > -1) {
			foreach ($this->groups as $key => $value) {
				$group .= $value['route'];
			}
		}
		$path = dirname($_SERVER['PHP_SELF']);
		$path = $path === '/' || strpos($this->runningPath, $path) !== 0 ? '' : $path;
		if (strstr($path, 'index.php')) {
			$data = implode('/', explode('/', $path));
			$path = str_replace($data, '', $path);
		}
		$route = $path . $group . '/' . trim($uri, '/');
		$route = rtrim($route, '/');
		if ($route === $path) {
			$route .= '/';
		}
		$routeName = is_string($callback)
			? strtolower(preg_replace(
				'/[^\w]/i', '.', str_replace($this->namespaces['controller'], '', $callback)
			))
			: NULL;
		$data = [
			'route'    => $this->clearRouteName($route),
			'method'   => strtoupper($method),
			'callback' => $callback,
			'name'     => isset($settings['name']) ? $settings['name'] : $routeName,
			'before'   => isset($settings['before']) ? $settings['before'] : NULL,
			'after'    => isset($settings['after']) ? $settings['after'] : NULL,
			'group'    => $groupItem === -1 ? NULL : $this->groups[$groupItem],
		];
		array_push($this->routes, $data);
	}

	/**
	 * Add new route method one or more http methods.
	 *
	 * @param string $methods
	 * @param string $route
	 * @param array|string|closure $settings
	 * @param string|closure $callback
	 *
	 * @return bool
	 */
	public function add($methods, $route, $settings, $callback = NULL) {
		if ($this->cacheLoaded) {
			return true;
		}
		if (is_null($callback)) {
			$callback = $settings;
			$settings = NULL;
		}
		if (strstr($methods, '|')) {
			foreach (array_unique(explode('|', $methods)) as $method) {
				if (!empty($method)) {
					call_user_func_array([
						$this,
						strtolower($method)
					], [
						$route,
						$settings,
						$callback
					]);
				}
			}
		} else {
			call_user_func_array([
				$this,
				strtolower($methods)
			], [
				$route,
				$settings,
				$callback
			]);
		}
		return true;
	}

	/**
	 * Add new route rules pattern; String or Array
	 *
	 * @param string|array $pattern
	 * @param null|string $attr
	 *
	 * @return mixed
	 * @throws
	 */
	public function pattern($pattern, $attr = NULL) {
		if (is_array($pattern)) {
			foreach ($pattern as $key => $value) {
				if (in_array($key, array_keys($this->patterns))) {
					return $this->exception($key . ' pattern cannot be changed.');
				}
				$this->patterns[$key] = '(' . $value . ')';
			}
		} else {
			if (in_array($pattern, array_keys($this->patterns))) {
				return $this->exception($pattern . ' pattern cannot be changed.');
			}
			$this->patterns[$pattern] = '(' . $attr . ')';
		}
		return true;
	}

	/**
	 * Run Routes
	 *
	 * @return void
	 * @throws
	 */
	public function __destruct() {
		if ($this->outputSent) {
			return;
		}
		$base = str_replace('\\', '/', str_replace($this->documentRoot, '', $this->runningPath));
		$uri = rtrim(parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH), '/');
		if ($_SERVER['REQUEST_URI'] !== $_SERVER['PHP_SELF']) {
			$uri = str_replace((dirname($_SERVER['PHP_SELF']) !== '/' ? dirname($_SERVER['PHP_SELF']) : ''), '', $uri);
		}
		if (($base !== $uri) && (substr($uri, -1) === '/')) {
			$uri = substr($uri, 0, (strlen($uri) - 1));
		}
		$uri = $this->clearRouteName($uri);
		$method = RouterRequest::getRequestMethod();
		$searches = array_keys($this->patterns);
		$replaces = array_values($this->patterns);
		$foundRoute = false;
		$routes = array_column($this->routes, 'route');
		// check if route is defined without regex
		if (in_array($uri, $routes)) {
			$currentRoute = array_filter($this->routes, function ($r) use ($method, $uri) {
				return RouterRequest::validMethod($r['method'], $method) && $r['route'] === $uri;
			});
			if (!empty($currentRoute)) {
				$currentRoute = current($currentRoute);
				$foundRoute = true;
				$this->runRouteMiddleware($currentRoute, 'before');
				$this->runRouteCommand($currentRoute['callback']);
				$this->runRouteMiddleware($currentRoute, 'after');
			}
		} else {
			foreach ($this->routes as $data) {
				$route = $data['route'];
				if (strstr($route, ':') !== false or (strstr($route, '{') !== false and strstr($route, '}') !== false)) {
					$route = str_replace($searches, $replaces, $route);
				}
				if (preg_match('#^' . $route . '$#', $uri, $matched)) {
					if (RouterRequest::validMethod($data['method'], $method)) {
						$foundRoute = true;
						$this->runRouteMiddleware($data, 'before', $matched);
						array_shift($matched);
						$matched = array_map(function ($value) {
							return trim(urldecode($value));
						}, $matched);
						$this->runRouteCommand($data['callback'], $matched);
						$this->runRouteMiddleware($data, 'after');
						break;
					}
				}
			}
		}
		// If it originally was a HEAD request, clean up after ourselves by emptying the output buffer
		if (strtoupper($_SERVER['REQUEST_METHOD']) === 'HEAD') {
			ob_end_clean();
		}
		if ($foundRoute === false) {
			if (!$this->errorCallback) {
				$this->errorCallback = function () {
					return $this->exception('Route not found. Looks like something went wrong. Please try again.');
				};
			}
			call_user_func($this->errorCallback);
		}
	}

	/**
	 * Detect Routes Middleware; before or after
	 *
	 * @param $middleware
	 * @param $type
	 *
	 * @return void
	 */
	public function runRouteMiddleware($middleware, $type, $matched = NULL) {
		if ($type === 'before') {
			if (!is_null($middleware['group'])) {
				$this->routerCommand()->beforeAfter($middleware['group'][$type], $matched);
			}
			$this->routerCommand()->beforeAfter($middleware[$type], $matched);
		} else {
			$this->routerCommand()->beforeAfter($middleware[$type], $matched);
			if (!is_null($middleware['group'])) {
				$this->routerCommand()->beforeAfter($middleware['group'][$type], $matched);
			}
		}
	}

	/**
	 * RouterCommand class
	 *
	 * @return RouterCommand
	 */
	public function routerCommand() {
		return RouterCommand::getInstance($this->baseFolder, $this->paths, $this->namespaces);
	}

	/**
	 * Run Route Command; Controller or Closure
	 *
	 * @param $command
	 * @param $params
	 *
	 * @return void
	 */
	private function runRouteCommand($command, $params = NULL) {
		$this->routerCommand()->runRoute($command, $params);
	}

	/**
	 * Added route from methods of Controller file.
	 *
	 * @param string $route
	 * @param string|array $settings
	 * @param null|string $controller
	 *
	 * @return mixed
	 * @throws
	 */
	public function controller($route, $settings, $controller = NULL) {
		if ($this->cacheLoaded) {
			return true;
		}
		if (is_null($controller)) {
			$controller = $settings;
			$settings = [];
		}
		$controller = $this->resolveClass($controller);
		$classMethods = get_class_methods($controller);
		if ($classMethods) {
			foreach ($classMethods as $methodName) {
				if (!strstr($methodName, '__')) {
					$method = 'any';
					foreach (explode('|', RouterRequest::$validMethods) as $m) {
						if (stripos($methodName, strtolower($m), 0) === 0) {
							$method = strtolower($m);
							break;
						}
					}
					$methodVar = lcfirst(preg_replace('/' . $method . '/i', '', $methodName, 1));
					$methodVar = strtolower(preg_replace('%([a-z]|[0-9])([A-Z])%', '\1-\2', $methodVar));
					$r = new ReflectionMethod($controller, $methodName);
					$endpoints = [];
					foreach ($r->getParameters() as $param) {
						$pattern = ':any';
						$typeHint = $param->hasType() ? $param->getType()->getName() : NULL;
						if (in_array($typeHint, [
							'int',
							'bool'
						])) {
							$pattern = ':id';
						} elseif (in_array($typeHint, [
							'string',
							'float'
						])) {
							$pattern = ':slug';
						} elseif ($typeHint === NULL) {
							$pattern = ':any';
						} else {
							continue;
						}
						$endpoints[] = $param->isOptional() ? $pattern . '?' : $pattern;
					}
					$value = ($methodVar === $this->mainMethod ? $route : $route . '/' . $methodVar);
					$this->{$method}(
						($value . '/' . implode('/', $endpoints)),
						$settings,
						($controller . '@' . $methodName)
					);
				}
			}
			unset($r);
		}
		return true;
	}

	/**
	 * @param $controller
	 *
	 * @return RouterException|mixed
	 */
	protected function resolveClass($controller) {
		$controller = str_replace([
			'\\',
			'.'
		], '/', $controller);
		$controller = trim(
			preg_replace(
				'/' . str_replace('/', '\\/', $this->paths['controllers']) . '/i',
				'', $controller,
				1
			),
			'/'
		);
		$file = realpath(rtrim($this->paths['controllers'], '/') . '/' . $controller . '.php');
		if (!file_exists($file)) {
			return $this->exception($controller . ' class is not found!');
		}
		$controller = $this->namespaces['controller'] . str_replace('/', '\\', $controller);
		if (!class_exists($controller)) {
			require $file;
		}
		return $controller;
	}

	/**
	 * Display all Routes.
	 *
	 * @return void
	 */
	public function getList() {
		echo '<pre>';
		var_dump($this->getRoutes());
		echo '</pre>';
		die;
	}

	/**
	 * Get all Routes
	 *
	 * @return mixed
	 */
	public function getRoutes() {
		return $this->routes;
	}

	/**
	 * Cache all routes
	 *
	 * @return bool
	 * @throws Exception
	 */
	public function cache() {
		foreach ($this->getRoutes() as $key => $r) {
			if (!is_string($r['callback'])) {
				throw new Exception(sprintf('Routes cannot contain a Closure/Function callback while caching.'));
			}
		}
		$cacheContent = '<?php return ' . var_export($this->getRoutes(), true) . ';' . PHP_EOL;
		if (false === file_put_contents($this->cacheFile, $cacheContent)) {
			throw new Exception(sprintf('Routes cache file could not be written.'));
		}
		return true;
	}

	public function use($class) {
		if ($class instanceof ApiVersions) {
			$classes = $class->getClasses();
			foreach ($classes as $class) {
				$this->importClass($class);
			}
		} else {
			$this->importClass($class);
		}
	}

	private function importClass(string $class) {
		$namespaceAndClass = explode('\\', $class);
		if ($namespaceAndClass[0] == 'routes') {
			include __API__ . $namespaceAndClass[1] . '/' . $namespaceAndClass[2] . '.php';
			$instance = new $class();
			if ($instance instanceof BaseRouter) {
				$this->group('/' . $instance->version . '/' . $instance->groupPath, function (Router $router) use ($instance) {
					$instance->routes($router);
				});
			}
		}
	}
}
