<?php

namespace routes\v2;

use Api\BaseRouter;
use FwRoutingSystem\Router;
use model\Artists;
use model\Comments;
use model\Entity\ArtistsEntity;
use model\Entity\AudioBooksEntity;
use model\Entity\BookmarksEntity;
use model\Entity\CommentsEntity;
use model\Entity\LabelsEntity;
use model\Entity\MembershipsEntity;
use model\Entity\PoemCategoriesEntity;
use model\Entity\PoemsEntity;
use model\Entity\PoetsEntity;
use model\Entity\StoriesEntity;
use model\Entity\TimePeriodsEntity;
use model\Entity\UsersEntity;
use model\Labels;
use model\Memberships;
use model\PoemCategories;
use model\Poems;
use model\Poets;
use model\Stories;
use model\TimePeriods;
use model\Users;
use PDO;
use version\ApiVersions;

class  ContentRoute extends BaseRouter {

	public string $version = ApiVersions::two;
	public $groupPath = 'content';

	public function routes(Router $router) {
		$router->group('/poem', function (Router $router) {
			$router->get('/poet/:id', function ($poetId) {
				$poet = Poets::get($poetId);
				if ($poet instanceof PoetsEntity) {
					$categories = PoemCategories::Db()->where('poet_id', $poet->poet_id)->where('parent_id', 0)->get()->map(function (PoemCategoriesEntity $categoriesEntity) {
						return $categoriesEntity->apiFormat();
					})->values();
					return response([
						'poet'       => $poet->apiFormat(),
						'categories' => $categories,
					]);
				}
				return response(['message' => "شاعر یافت نشد"], 404);
			});
			$router->get('/category/:id', function ($id) {
				$category = PoemCategories::get($id);
				if ($category instanceof PoemCategoriesEntity) {
					$categories = PoemCategories::Db()->where('parent_id', $category->poem_category_id)->get()->map(function (PoemCategoriesEntity $categoriesEntity) {
						return $categoriesEntity->apiFormat();
					})->values();
					return response([
						'category'   => $category->apiFormat(),
						'categories' => $categories->values(),
					]);
				}
				return response(['message' => "دسته بندی یافت نشد"], 404);
			});
			$router->post('/poems/:id', function ($id) {
				$category = PoemCategories::get($id);
				if ($category instanceof PoemCategoriesEntity) {
					$perPage = 6; // Number of poems per page
					$currentPage = $this->getParam('page');
					if ($currentPage == 0){
						$currentPage = 1;
					}
					$query = $this->getParam('query');

					// Fetch paginated poems
					$poemsQuery = Poems::Db()->where('poem_category_id', $category->poem_category_id);
					$total = Poems::Db()->where('poem_category_id', $category->poem_category_id)->rowCount();
					if ($query != "") {
						$poemsQuery = $poemsQuery->where('poem_text', 'like', "%$query%");
					}
					$poems = $poemsQuery->offset(($currentPage - 1) * $perPage)->limit($perPage)->get()->map(function (PoemsEntity $poemsEntity) {
						return $poemsEntity->apiFormat();
					})->values();

					// Create pagination metadata
					$pagination = [
						'current_page' => (int)$currentPage,
						'per_page'     => (int)$perPage,
						'total'        => (int)$total,
						'last_page'    => ceil($total / $perPage),
					];

					return response([
						'category'   => $category->apiFormat(),
						'poems'      => $poems->values(),
						'pagination' => $pagination,
					]);
				}
				return response(['message' => "دسته بندی یافت نشد"], 404);
			});
			$router->get('/poem/:id', function ($poemId) {
				$poem = Poems::get($poemId);
				if ($poem instanceof PoemsEntity) {
					$category = PoemCategories::get($poem->category_id);
					$categories = [];
					$categories = $this->fetchCategories($category, $categories);

					$poet = Poets::get($poem->poet_id);
					return response([
						'categories' => $categories,
						'poem'       => $poem->singleFormat(),
						'poet'       => $poet->apiFormat(),
					]);
				}
				return response(['message' => "شاعر یافت نشد"], 404);
			});

		});
		$router->post('/comments/:id?', function ($episodeId = 0) {
			$content_id = $this->getParam('content_id');
			$type = $this->getParam('type');
			$column = "";
			switch ($type) {
				case "story":
					$column = "story_id";
					break;
				case "poem":
					$column = "poem_id";
					break;
				case "book":
					$column = "book_id";
					break;
				case "audioBook":
					$column = "audio_book_id";
					break;
			}
			return response(Comments::Db()->where([
				$column      => $content_id,
				'episode_id' => $episodeId,
			])->orderBy('created_at', true)->get()->map(function (CommentsEntity $commentsEntity) {
				return $commentsEntity->apiFormat();
			})->values());
		});
		$router->get('/audioBook/:id', function ($id) {
			$audio = \model\AudioBooks::get($id);
			if ($audio instanceof AudioBooksEntity) {
				$bookmark = null;
				$user = Users::findToken(get_header('auth'));

				$hasAccess = false;
				$userId = 0;
				if ($user instanceof UsersEntity) {
					$userId = $user->user_id;
					$bookmark = \model\Bookmarks::Db()->where('user_id', $user->user_id)->where('audio_book_id', $id)->get()->first;
					$plan = Memberships::get($user->membership_id);
					$hasAccess = !(($plan instanceof MembershipsEntity and $user->membership_expired_at > 0 and $user->membership_activated_at > 0 and $user->membership_expired_at <= time()) or $plan == null);
				}

				return response(['audioBook' => $audio->singleApiFormat($bookmark instanceof BookmarksEntity, $hasAccess, $userId),]);

			}
			return response(['message' => "کتاب صوتی یافت نشد"], 404);

		});

		$router->get('/story/:id', function ($id) {

			$story = Stories::get($id);
			if ($story instanceof StoriesEntity) {
				$bookmark = null;
				$user = Users::findToken(get_header('auth'));

				$hasAccess = false;
				$userId = 0;
				if ($user instanceof UsersEntity) {
					$userId = $user->user_id;
					$bookmark = \model\Bookmarks::Db()->where('user_id', $user->user_id)->where('story_id', $id)->get()->first;

					$plan = Memberships::get($user->membership_id);
					$hasAccess = !(($plan instanceof MembershipsEntity and $user->membership_expired_at > 0 and $user->membership_activated_at > 0 and $user->membership_expired_at <= time()) or $plan == null);
				}
				return response(['story' => $story->singleApiFormat($bookmark instanceof BookmarksEntity, $hasAccess, $userId),]);
			}
			return response(['message' => "داستان یافت نشد"], 404);
		});

		$router->post('/search', function () {
			$user = Users::findToken(get_header('auth'));
			$query = $this->getParam('q');
			$type = $this->getParam('type');
			$artists = json_decode($this->getParam('artists'), true); // Decoding artists filter
			$tags = json_decode($this->getParam('tags'), true); // Decoding tags filter
			$sortType = json_decode($this->getParam('sortType'), true); // Decoding sort type
			$page = (int)$this->getParam('page', 1); // Default page is 1
			$perPage = (int)$this->getParam('perPage', 10); // Default items per page is 10

			$pdo = \FwConnection::conn();
			switch ($type) {
				case 1: // Stories
					return response($this->searchStories($pdo, $query, $artists, $tags, $sortType, $page, $perPage));
				case 2: // AudioBooks
					return response($this->searchAudioBooks($pdo, $query, $artists, $tags, $sortType, $page, $perPage));
				case 5: // Poems
					// Apply sorting logic based on the last episode date
					return response($this->searchPoets($pdo, $query, $sortType));
				default:
					return response([
						'message' => "$type is not a valid type",
					], 404);
			}
		});


		$router->get('/artists', function () {
			$artists = Artists::Db()->get()->map(function (ArtistsEntity $artistsEntity) {
				return $artistsEntity->apiFormat();
			});
			return response($artists->values());
		});

		$router->get('/labels', function () {
			$labels = Labels::Db()->get()->map(function (LabelsEntity $labelsEntity) {
				return $labelsEntity->apiFormat();
			});
			return response($labels->values());
		});

	}

	private function fetchCategories($category, &$categories) {
		if ($category instanceof PoemCategoriesEntity) {
			$categories[] = $category->apiFormat();
			if ($category->parent_id > 0) {
				$parent = PoemCategories::get($category->parent_id);
				return $this->fetchCategories($parent, $categories);
			}
		}
		return $categories;
	}

	/**
	 * Search Stories
	 */
	function searchStories($pdo, $query, $artists, $tags, $sortType, $page, $perPage) {
		$offset = ($page - 1) * $perPage;

		// Base query with joins for like count, comment count, and last episode date
		$baseQuery = "SELECT tblStories.*,
                        (SELECT COUNT(*) FROM tblLikes WHERE tblLikes.story_id = tblStories.story_id) as like_count,
                        (SELECT COUNT(*) FROM tblComments WHERE tblComments.story_id = tblStories.story_id) as comment_count,
                        MAX(tblEpisodes.episode_date) as last_episode_date
                  FROM tblStories
                  LEFT JOIN tblEpisodes ON tblEpisodes.story_id = tblStories.story_id
                  WHERE story_name LIKE ?";
		$bindings = ['%' . $query . '%'];

		// Apply artist filter
		if (!empty($artists)) {
			$artistPlaceholders = implode(',', array_fill(0, count($artists), '?'));
			$baseQuery .= " AND artist_id IN ($artistPlaceholders)";
			$bindings = array_merge($bindings, $artists);
		}

		// Apply tag filter
		if (!empty($tags)) {
			$tagPlaceholders = implode(',', array_fill(0, count($tags), '?'));
			$baseQuery .= " AND label_ids IN ($tagPlaceholders)";
			$bindings = array_merge($bindings, $tags);
		}

		// Group by story to ensure aggregate functions work correctly
		$baseQuery .= " GROUP BY tblStories.story_id";

		// Apply sorting logic based on the last episode
		switch ($sortType) {
			case 'به ترتیب الفبا (ا - ی)':
				$baseQuery .= " ORDER BY story_name ASC";
				break;
			case 'به ترتیب الفبا (ی - ا)':
				$baseQuery .= " ORDER BY story_name DESC";
				break;
			case 'محبوب‌ترین':
				$baseQuery .= " ORDER BY like_count DESC";
				break;
			case 'پر حاشیه‌ترین':
				$baseQuery .= " ORDER BY comment_count DESC";
				break;
			case 'جدید‌ترین':
				$baseQuery .= " ORDER BY last_episode_date DESC";
				break;
			case 'قدیمی‌ترین':
				$baseQuery .= " ORDER BY last_episode_date ASC";
				break;
			default:
				$baseQuery .= " ORDER BY last_episode_date DESC";
		}

		// Add pagination
		$baseQuery .= " LIMIT ? OFFSET ?";
		$bindings[] = $perPage;
		$bindings[] = $offset;

		// Execute query
		$stmt = $pdo->prepare($baseQuery);
		$stmt->execute($bindings);

		// Initialize the stories array
		$stories = [];

		// Fetch each row as a StoriesEntity object
		while ($row = $stmt->fetchObject(StoriesEntity::class)) {
			$stories[] = $row->apiFormat();
		}

		return [
			'stories'    => $stories,
			'pagination' => [
				'page'    => $page,
				'perPage' => $perPage,
				'total'   => $this->getTotalCount($pdo, 'tblStories', 'story_name LIKE ?', ['%' . $query . '%']),
			],
		];
	}

	/**
	 * Get total count of rows matching a condition
	 */
	function getTotalCount($pdo, $table, $condition, $bindings) {
		$stmt = $pdo->prepare("SELECT COUNT(*) as total FROM $table WHERE $condition");
		$stmt->execute($bindings);
		$result = $stmt->fetch(PDO::FETCH_ASSOC);
		return $result['total'] ?? 0;
	}

	/**
	 * Search AudioBooks
	 */
	function searchAudioBooks($pdo, $query, $artists, $tags, $sortType, $page, $perPage): array {
		$offset = ($page - 1) * $perPage;

		// Base query with joins for like count, comment count, and last episode date
		$baseQuery = "SELECT tblAudioBooks.*,
                        (SELECT COUNT(*) FROM tblLikes WHERE tblLikes.audio_book_id = tblAudioBooks.audio_book_id) as like_count,
                        (SELECT COUNT(*) FROM tblComments WHERE tblComments.audio_book_id = tblAudioBooks.audio_book_id) as comment_count,
                        MAX(tblAudioBookEpisodes.episode_date) as last_episode_date
                  FROM tblAudioBooks
                  LEFT JOIN tblAudioBookEpisodes ON tblAudioBookEpisodes.audio_book_id = tblAudioBooks.audio_book_id
                  WHERE audio_book_name LIKE ?";
		$bindings = ['%' . $query . '%'];

		// Apply artist filter
		if (!empty($artists)) {
			$artistPlaceholders = implode(',', array_fill(0, count($artists), '?'));
			$baseQuery .= " AND artist_id IN ($artistPlaceholders)";
			$bindings = array_merge($bindings, $artists);
		}

		// Apply tag filter
		if (!empty($tags)) {
			$tagPlaceholders = implode(',', array_fill(0, count($tags), '?'));
			$baseQuery .= " AND tag_id IN ($tagPlaceholders)";
			$bindings = array_merge($bindings, $tags);
		}

		// Group by audio book to ensure aggregate functions work correctly
		$baseQuery .= " GROUP BY tblAudioBooks.audio_book_id";

		// Apply sorting logic based on the last episode date
		switch ($sortType) {
			case 'به ترتیب الفبا (ا - ی)':
				$baseQuery .= " ORDER BY audio_book_name ASC";
				break;
			case 'به ترتیب الفبا (ی - ا)':
				$baseQuery .= " ORDER BY audio_book_name DESC";
				break;
			case 'محبوب‌ترین':
				$baseQuery .= " ORDER BY like_count DESC";
				break;
			case 'پر حاشیه‌ترین':
				$baseQuery .= " ORDER BY comment_count DESC";
				break;
			case 'جدید‌ترین':
				$baseQuery .= " ORDER BY last_episode_date DESC";
				break;
			case 'قدیمی‌ترین':
				$baseQuery .= " ORDER BY last_episode_date ASC";
				break;
			default:
				$baseQuery .= " ORDER BY last_episode_date DESC";
		}

		// Add pagination
		$baseQuery .= " LIMIT ? OFFSET ?";
		$bindings[] = $perPage;
		$bindings[] = $offset;

		// Execute query
		$stmt = $pdo->prepare($baseQuery);
		$stmt->execute($bindings);

		// Initialize the audioBooks array
		$audioBooks = [];

		// Fetch each row as an AudioBookEntity object
		while ($row = $stmt->fetchObject(AudioBooksEntity::class)) {
			$audioBooks[] = $row->apiFormat();
		}

		return [
			'audioBooks' => $audioBooks,
			'pagination' => [
				'page'    => $page,
				'perPage' => $perPage,
				'total'   => $this->getTotalCount($pdo, 'tblAudioBooks', 'audio_book_name LIKE ?', ['%' . $query . '%']),
			],
		];
	}

	/**
	 * Search Poems
	 */
	function searchPoets($pdo, $query, $sortType): array {
		$baseQuery = TimePeriods::Db();
		switch ($sortType) {
			case 'جدید‌ترین':
				$baseQuery = $baseQuery->orderBy("time_period_id");
				break;
			case 'قدیمی‌ترین':
				$baseQuery = $baseQuery->orderBy("time_period_id", true);
				break;
			default:
				$baseQuery = $baseQuery->orderBy("time_period_id");
		}
		$timePeriods = $baseQuery->get()->map(function (TimePeriodsEntity $timePeriodsEntity) {
			return $timePeriodsEntity->apiFormat();
		})->values();

		$periods = [];
		foreach ($timePeriods as $period) {
			if (!isset($periods[$period['id']])) {
				$periods[$period['id']] = $period;
			}
			$periods[$period['id']]['poets'] = [
				...$periods[$period['id']]['poets'],
				...Poets::Db()->where('time_period_id', $period['id'])->get()->filter(function (PoetsEntity $poetsEntity) use ($query) {
					return !($query != "") || str($poetsEntity->name)->includes($query);
				})->map(function (PoetsEntity $poetsEntity) {
					return $poetsEntity->apiFormat();
				})->values(),
			];
		}
		array_unshift($periods, [
			'id'      => 0,
			'name'    => "شاعرین منتخب",
			'details' => "",
			'poets'   => Poets::Db()->whereIn('poet_id', [
				7,
				88,
				75,
				13,
				24,
				57,
			])->get()->filter(function (PoetsEntity $poetsEntity) use ($query) {
				return !($query != "") || str($poetsEntity->name)->includes($query);
			})->map(function (PoetsEntity $poetsEntity) {
				return $poetsEntity->apiFormat();
			})->values(),
		]);
		return collect(array_values($periods))->filter(function ($period) {
			return sizeof($period['poets']) > 0;
		})->values()->all();
	}

	public function requiredHeaders(): array {
		return [];
	}

}