<?php

namespace routes\v2;

use Api\BaseRouter;
use FwRoutingSystem\Router;
use model\Entity\BooksEntity;
use model\Users;
use PDO;
use version\ApiVersions;

class ShopRoute extends BaseRouter
{
	public string $version = ApiVersions::two;
	public $groupPath = 'shop';

	public function routes(Router $router)
	{
		$router->post('/search', function () {
			$user = Users::findToken(get_header('auth'));
			$query = $this->getParam('q');
			$artists = json_decode($this->getParam('artists'), true); // Decoding artists filter
			$tags = json_decode($this->getParam('tags'), true); // Decoding tags filter
			$sortType = json_decode($this->getParam('sortType'), true); // Decoding sort type
			$page = (int)$this->getParam('page', 1); // Default page is 1
			$perPage = (int)$this->getParam('perPage', 10); // Default items per page is 10

			$pdo = \FwConnection::conn();

			// Search books directly
			return response($this->searchBooks($pdo, $query, $artists, $tags, $sortType, $page, $perPage));
		});
	}

	public function requiredHeaders(): array
	{
		return [];
	}

	/**
	 * Search Books
	 */
	private function searchBooks($pdo, $query, $artists, $tags, $sortType, $page, $perPage): array
	{
		$offset = ($page - 1) * $perPage;

		// Base query
		$baseQuery = "SELECT tblBooks.*,
                        (SELECT COUNT(*) FROM tblLikes WHERE tblLikes.book_id = tblBooks.book_id) as like_count,
                        (SELECT COUNT(*) FROM tblComments WHERE tblComments.book_id = tblBooks.book_id) as comment_count
                  FROM tblBooks
                  WHERE book_name LIKE ?";
		$bindings = ['%' . $query . '%'];

		// Apply artist filter
		if (!empty($artists)) {
			$artistPlaceholders = implode(',', array_fill(0, count($artists), '?'));
			$baseQuery .= " AND artist_id IN ($artistPlaceholders)";
			$bindings = array_merge($bindings, $artists);
		}

		// Apply tag filter
		if (!empty($tags)) {
			$tagPlaceholders = implode(',', array_fill(0, count($tags), '?'));
			$baseQuery .= " AND label_ids IN ($tagPlaceholders)";
			$bindings = array_merge($bindings, $tags);
		}

		// Apply sorting logic
		switch ($sortType) {
			case 'alphabetical_asc':
				$baseQuery .= " ORDER BY book_name ASC";
				break;
			case 'alphabetical_desc':
				$baseQuery .= " ORDER BY book_name DESC";
				break;
			case 'most_popular':
				$baseQuery .= " ORDER BY like_count DESC";
				break;
			case 'most_commented':
				$baseQuery .= " ORDER BY comment_count DESC";
				break;
			case 'newest':
				$baseQuery .= " ORDER BY book_publish_date DESC";
				break;
			case 'oldest':
				$baseQuery .= " ORDER BY book_publish_date ASC";
				break;
			default:
				$baseQuery .= " ORDER BY book_publish_date DESC";
		}

		// Add pagination
		$baseQuery .= " LIMIT ? OFFSET ?";
		$bindings[] = $perPage;
		$bindings[] = $offset;

		// Execute query
		$stmt = $pdo->prepare($baseQuery);
		$stmt->execute($bindings);

		// Initialize the books array
		$books = [];

		// Fetch each row
		while ($row = $stmt->fetchObject(BooksEntity::class)) {
			$books[] = $row->apiFormat();
		}

		return [
			'books' => $books,
			'pagination' => [
				'page' => $page,
				'perPage' => $perPage,
				'total' => $this->getTotalCount($pdo, 'tblBooks', 'book_name LIKE ?', ['%' . $query . '%']),
			],
		];
	}

	/**
	 * Get total count of rows matching a condition
	 */
	private function getTotalCount($pdo, $table, $condition, $bindings)
	{
		$stmt = $pdo->prepare("SELECT COUNT(*) as total FROM $table WHERE $condition");
		$stmt->execute($bindings);
		$result = $stmt->fetch(PDO::FETCH_ASSOC);
		return $result['total'] ?? 0;
	}
}
